<?php
/**
 * OGP Analytics (cards + counters)
 * - Table: wp*_ogp_analytics (completed / cancelled / auto_cleanup)
 * - Counts:
 *   - Pending: from wp*_ogp_incomplete_checkout, BUT
 *       • exclude rows with empty products
 *       • exclude rows that match a real Woo order (Legacy or HPOS) by phone
 *   - Completed/Cancelled/Auto Cleanup: from wp*_ogp_analytics
 *     (UI actions only; Woo real order hook must NOT log events)
 */

if (!defined('ABSPATH')) exit; // ✅ security guard

add_action('admin_init', function () {
    global $wpdb;
    $table = $wpdb->prefix . 'ogp_analytics';

    // Create or update analytics table
    $exists = $wpdb->get_var( $wpdb->prepare("SHOW TABLES LIKE %s", $table) );
    if ($exists !== $table) {
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        $charset_collate = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE $table (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            lead_id bigint(20) UNSIGNED NOT NULL,
            phone varchar(20) NOT NULL,
            status enum('completed','cancelled','auto_cleanup') NOT NULL,
            created_at datetime NOT NULL,
            subtotal decimal(10,2) DEFAULT 0.00,
            ip_address varchar(45) DEFAULT '',
            PRIMARY KEY (id),
            KEY lead_id (lead_id),
            KEY phone (phone),
            KEY status (status),
            KEY created_at (created_at)
        ) $charset_collate;";
        dbDelta($sql);
    } else {
        // keep enum and columns synced
        $wpdb->query("ALTER TABLE $table MODIFY COLUMN status ENUM('completed','cancelled','auto_cleanup') NOT NULL");

        $need = ['lead_id','phone','status','created_at','subtotal','ip_address'];
        foreach ($need as $col) {
            $has = $wpdb->get_var( $wpdb->prepare("SHOW COLUMNS FROM $table LIKE %s", $col) );
            if (!$has) {
                if ($col === 'status') {
                    $wpdb->query("ALTER TABLE $table ADD COLUMN status ENUM('completed','cancelled','auto_cleanup') NOT NULL AFTER phone");
                } elseif ($col === 'created_at') {
                    $wpdb->query("ALTER TABLE $table ADD COLUMN created_at DATETIME NOT NULL AFTER status");
                } elseif ($col === 'subtotal') {
                    $wpdb->query("ALTER TABLE $table ADD COLUMN subtotal DECIMAL(10,2) DEFAULT 0.00 AFTER created_at");
                } elseif ($col === 'ip_address') {
                    $wpdb->query("ALTER TABLE $table ADD COLUMN ip_address VARCHAR(45) DEFAULT '' AFTER subtotal");
                } elseif ($col === 'lead_id') {
                    $wpdb->query("ALTER TABLE $table ADD COLUMN lead_id BIGINT(20) UNSIGNED NOT NULL AFTER id");
                } elseif ($col === 'phone') {
                    $wpdb->query("ALTER TABLE $table ADD COLUMN phone VARCHAR(20) NOT NULL AFTER lead_id");
                }
            }
        }
    }
});

/** UI actions write here (Save/Delete/Cron only) */
function ogp_ic_log_event($lead_id, $phone, $status, $subtotal = 0, $ip = '') {
    global $wpdb;
    $table  = $wpdb->prefix . 'ogp_analytics';
    $status = in_array($status, ['completed','cancelled','auto_cleanup'], true) ? $status : 'completed';

    $wpdb->insert(
        $table,
        [
            'lead_id'    => (int) $lead_id,
            'phone'      => substr((string) $phone, 0, 20),
            'status'     => $status,
            'created_at' => current_time('mysql'),
            'subtotal'   => (float) $subtotal,
            'ip_address' => substr((string) $ip, 0, 45),
        ],
        ['%d','%s','%s','%s','%f','%s']
    );
}

function ogp_ic_table_exists($table){
    global $wpdb;
    return $wpdb->get_var( $wpdb->prepare("SHOW TABLES LIKE %s", $table) ) === $table;
}

/**
 * Count "Pending" (true incomplete only) within time range.
 * Rules:
 *  - products must NOT be empty
 *  - exclude if a real Woo order exists for same phone (Legacy posts or HPOS)
 */
function ogp_ic_count_incomplete_between($from, $to){
    global $wpdb;
    $ic = $wpdb->prefix . 'ogp_incomplete_checkout';
    if (!ogp_ic_table_exists($ic)) return 0;

    $statuses = array(
        'wc-pending','wc-processing','wc-on-hold',
        'wc-completed','wc-cancelled','wc-refunded','wc-failed'
    );

    // Legacy posts + postmeta
    $posts = $wpdb->posts;
    $pm    = $wpdb->postmeta;

    // normalize phone both sides (last 10 digits after stripping + - space)
    $norm_ic   = "RIGHT(REPLACE(REPLACE(REPLACE(ic.phone,'+',''),'-',''),' ',''),10)";
    $norm_meta = "RIGHT(REPLACE(REPLACE(REPLACE(pm.meta_value,'+',''),'-',''),' ',''),10)";

    // HPOS availability?
    $hpos_orders  = $wpdb->prefix . 'wc_orders';
    $hpos_address = $wpdb->prefix . 'wc_order_addresses';
    $hpos_exists  = $wpdb->get_var("SHOW TABLES LIKE '{$hpos_orders}'");

    $sql = "
        SELECT COUNT(*)
        FROM {$ic} ic
        WHERE ic.status = 'Pending'
          AND TRIM(ic.products) <> ''                /* empty products => exclude */
          AND ic.created_at BETWEEN %s AND %s
          AND NOT EXISTS (                            /* Legacy orders */
                SELECT 1
                FROM {$posts} p
                INNER JOIN {$pm} pm
                        ON pm.post_id = p.ID AND pm.meta_key = '_billing_phone'
                WHERE {$norm_meta} = {$norm_ic}
                  AND p.post_type   = 'shop_order'
                  AND p.post_status IN (" . implode(',', array_fill(0, count($statuses), '%s')) . ")
                  AND p.post_date  >= ic.created_at   /* ✅ use same field as above */
          )";

    $params = array($from, $to);
    foreach ($statuses as $s) { $params[] = $s; }

    if ($hpos_exists) {
        $sql .= " AND NOT EXISTS (                      /* HPOS orders */
                    SELECT 1
                    FROM {$hpos_orders} o
                    INNER JOIN {$hpos_address} a
                            ON a.order_id = o.id AND a.address_type = 'billing'
                    WHERE RIGHT(REPLACE(REPLACE(REPLACE(a.phone,'+',''),'-',''),' ',''),10) = {$norm_ic}
                      AND o.status IN (" . implode(',', array_fill(0, count($statuses), '%s')) . ")
                      AND o.date_created_gmt >= ic.created_at
                 )";
        foreach ($statuses as $s) { $params[] = $s; }
    }

    return (int) $wpdb->get_var( $wpdb->prepare($sql, $params) );
}

/** Aggregate counts in a time window */
function ogp_ic_get_stats_between($from, $to){
    global $wpdb;
    $events = $wpdb->prefix . 'ogp_analytics';

    $completed = 0;
    $cancelled = 0;
    $auto_cleanup = 0;

    if (ogp_ic_table_exists($events)) {
        $completed = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM $events WHERE status='completed' AND created_at BETWEEN %s AND %s",
            $from, $to
        ));
        $cancelled = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM $events WHERE status='cancelled' AND created_at BETWEEN %s AND %s",
            $from, $to
        ));
        $auto_cleanup = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM $events WHERE status='auto_cleanup' AND created_at BETWEEN %s AND %s",
            $from, $to
        ));
    }

    $pending = ogp_ic_count_incomplete_between($from, $to);

    $total = $completed + $cancelled + $auto_cleanup + $pending;
    $cr = $total > 0 ? round(($completed / $total) * 100, 1) : 0;

    return [
        'total'        => $total,
        'completed'    => $completed,
        'cancelled'    => $cancelled,
        'auto_cleanup' => $auto_cleanup,
        'pending'      => $pending,
        'cr'           => $cr,
    ];
}

/** Cards data for 24h / 7d / 30d / all */
function ogp_ic_analytics_cards_data(){
    $now = current_time('timestamp');
    $fmt = 'Y-m-d H:i:s';
    $to  = date_i18n($fmt, $now, false);

    $d24_from = date_i18n($fmt, $now - 24*3600, false);
    $d7_from  = date_i18n($fmt, $now - 7*24*3600, false);
    $d30_from = date_i18n($fmt, $now - 30*24*3600, false);
    $all_from = '1970-01-01 00:00:00';

    return [
        '24h' => ogp_ic_get_stats_between($d24_from, $to),
        '7d'  => ogp_ic_get_stats_between($d7_from,  $to),
        '30d' => ogp_ic_get_stats_between($d30_from, $to),
        'all' => ogp_ic_get_stats_between($all_from, $to),
    ];
}

/** Render a single analytics card (kept design as-is) */
function ogp_ic_render_card($title, $S){
    echo '<div style="border:1px solid #dcdcdc;border-radius:8px;padding:12px;background:#fff;">';
    echo '<h2 style="margin-top:0">' . esc_html($title) . '</h2>';
    echo '<p><strong>Total Tracked:</strong> ' . esc_html($S['total']) . '</p>';
    echo '<p><span style="color:green">✅ Completed:</span> ' . esc_html($S['completed']) . '</p>';
    echo '<p><span style="color:#c60">❌ Pending:</span> ' . esc_html($S['pending']) . '</p>';
    echo '<p><span style="color:#aa0000">✖️ Cancelled:</span> ' . esc_html($S['cancelled']) . '</p>';
    echo '<p><span style="color:#6b7280">🗑️ Auto Cleanup:</span> ' . esc_html($S['auto_cleanup']) . '</p>';
    echo '<p><strong>Conversion Rate:</strong> ' . esc_html($S['cr']) . '%</p>';
    echo '</div>';
}
