<?php
/**
 * License Handler for Incomplete Checkouts
 */
if (!defined('ABSPATH')) exit;

// =====================
// Lightweight debug logs (only if WP_DEBUG true)
// =====================
// Tiny helpers for safe logging & masking
if (!function_exists('ogp_log')) {
    function ogp_log($msg){
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[OGP] ' . $msg);
        }
    }
}
if (!function_exists('ogp_mask')) {
    function ogp_mask($s){
        return is_string($s) && strlen($s) > 4 ? (substr($s,0,2).'****'.substr($s,-2)) : $s;
    }
}

class OGP_IC_License_Handler {
    private $license_option = 'ogp_ic_license';
    private $api_url        = 'https://order-growth-plugin-license.inventrix-it.com/license-activations.php?action=';
    private $product_slug   = 'incomplete-checkouts';
    
    public function __construct() {
        add_action('admin_init', [$this, 'process_license_actions']);

        if (!wp_next_scheduled('ogp_ic_license_check')) {
            wp_schedule_event(time(), 'daily', 'ogp_ic_license_check');
        }
        add_action('ogp_ic_license_check', [$this, 'check_license_status']);
    }

    public function process_license_actions() {
        if (!current_user_can('manage_options')) return;

        if (isset($_POST['ogp_ic_license_activate'])) {
            if (!check_admin_referer('ogp_ic_license_nonce','ogp_ic_license_nonce')) return;
            $license_key = sanitize_text_field($_POST['ogp_ic_license_key']);
            $this->activate_license($license_key);
        }

        if (isset($_POST['ogp_ic_license_deactivate'])) {
            if (!check_admin_referer('ogp_ic_license_nonce','ogp_ic_license_nonce')) return;
            $this->deactivate_license();
        }
    }

    /**
     * Safe extractor for expires from API object (handles both expires_on and license->expires_at)
     */
    private function safe_expires($license_data) {
        if (is_object($license_data) && isset($license_data->expires_on) && is_string($license_data->expires_on)) {
            return $license_data->expires_on;
        }
        if (
            is_object($license_data)
            && isset($license_data->license)
            && is_object($license_data->license)
            && isset($license_data->license->expires_at)
            && is_string($license_data->license->expires_at)
        ) {
            return $license_data->license->expires_at;
        }
        return null;
    }

    private function activate_license($license_key) {
        $site_url = home_url();
        $instance = md5($site_url . get_bloginfo('name'));

        // Log call intent
        ogp_log('activate → site=' . $site_url . ' inst=' . $instance . ' key=' . ogp_mask($license_key));

        $response = wp_remote_post($this->api_url . 'activate', [
            'timeout' => 5, // 15 -> 5
            'body' => [
                'license_key'  => $license_key,
                'site_url'     => $site_url,
                'instance'     => $instance,
                'product_slug' => $this->product_slug,
                'platform'     => 'WordPress ' . get_bloginfo('version')
            ]
        ]);

        // Error FIRST, then read body
        if (is_wp_error($response)) {
            ogp_log('activate → API error: ' . $response->get_error_message());
            add_settings_error(
                'ogp_ic_license',
                'ogp_ic_license_error',
                'Error connecting to license server: ' . $response->get_error_message(),
                'error'
            );
            return;
        }

        $code          = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        ogp_log('activate → http=' . $code . ' body=' . substr($response_body,0,300));

        $license_data = json_decode($response_body);
        if (!is_object($license_data) || !isset($license_data->success)) {
            add_settings_error(
                'ogp_ic_license',
                'ogp_ic_license_error',
                'Invalid response from license server.',
                'error'
            );
            return;
        }

        if ($license_data->success) {
            $expires_from_api = $this->safe_expires($license_data);

            update_option($this->license_option, [
                'key'        => $license_key,
                'status'     => 'valid',
                'expires'    => $expires_from_api,
                'instance'   => $instance,
                'last_check' => time()
            ]);

            add_settings_error(
                'ogp_ic_license',
                'ogp_ic_license_activated',
                'License activated successfully!',
                'success'
            );
        } else {
            add_settings_error(
                'ogp_ic_license',
                'ogp_ic_license_error',
                'License activation failed: ' . (isset($license_data->message) ? $license_data->message : 'Unknown error'),
                'error'
            );
        }
    }

    private function deactivate_license() {
        $license = get_option($this->license_option);
        if (!$license || empty($license['key'])) {
            add_settings_error(
                'ogp_ic_license',
                'ogp_ic_license_error',
                'No license key found to deactivate.',
                'error'
            );
            return;
        }

        ogp_log('deactivate → key=' . ogp_mask($license['key']) . ' inst=' . ($license['instance'] ?? ''));

        wp_remote_post($this->api_url . 'deactivate', [
            'timeout' => 5, // 15 -> 5
            'body' => [
                'license_key'  => $license['key'],
                'instance'     => $license['instance'],
                'product_slug' => $this->product_slug
            ]
        ]);

        update_option($this->license_option, [
            'key'        => '',
            'status'     => 'invalid',
            'expires'    => null,
            'instance'   => '',
            'last_check' => time()
        ]);

        add_settings_error(
            'ogp_ic_license',
            'ogp_ic_license_deactivated',
            'License deactivated successfully.',
            'success'
        );
    }

    public function is_license_valid() {
        $license = get_option($this->license_option);
        if (!$license || empty($license['key']) || $license['status'] !== 'valid') return false;

        if (!empty($license['expires']) && strtotime($license['expires']) < time()) {
            update_option($this->license_option, array_merge($license, ['status' => 'expired']));
            return false;
        }
        return true;
    }

    /**
     * Daily cron validator
     * - API error / bad JSON => soft-fail (status অপরিবর্তিত)
     * - success:false + "bound to another domain" => status valid-ই থাকে
     * - success:false (অন্যান্য কারণ) => status = invalid
     */
    public function check_license_status() {
        $license = get_option($this->license_option);
        if (!$license || empty($license['key']) || $license['status'] !== 'valid') {
            return;
        }

        ogp_log('validate → key=' . ogp_mask($license['key']) . ' inst=' . ($license['instance'] ?? ''));

        $response = wp_remote_post($this->api_url . 'validate', [
            'timeout' => 5,
            'body' => [
                'license_key'  => $license['key'],
                'instance'     => $license['instance'],
                'product_slug' => $this->product_slug,
                'site_url'     => home_url(),
            ]
        ]);

        // Network / HTTP client error => never invalidate
        if (is_wp_error($response)) {
            ogp_log('validate → API error: ' . $response->get_error_message());
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        ogp_log('validate → http=' . $code . ' body=' . substr($body, 0, 300));

        $license_data = json_decode($body);
        if (!is_object($license_data) || !isset($license_data->success)) {
            ogp_log('validate → bad JSON or missing "success"');
            return;
        }

        // যদি success=false হয়, আগে message চেক করি
        if (!$license_data->success) {
            $msg = '';
            if (isset($license_data->message) && is_string($license_data->message)) {
                $msg = strtolower($license_data->message);
            }

            // License is bound to another domain → লোকাল লাইসেন্স invalid করবো না
            if (strpos($msg, 'bound to another domain') !== false) {
                ogp_log('validate → server says bound-to-other-domain – keeping license VALID.');

                $expires_from_api = $this->safe_expires($license_data);
                $new_expires      = $expires_from_api ?: ($license['expires'] ?? null);

                update_option($this->license_option, [
                    'key'        => $license['key'],
                    'status'     => 'valid',
                    'expires'    => $new_expires,
                    'instance'   => $license['instance'],
                    'last_check' => time(),
                ]);
                return;
            }

            // অন্য যেকোনো ক্ষেত্রে success:false => invalid
            update_option($this->license_option, [
                'key'        => $license['key'],
                'status'     => 'invalid',
                'expires'    => null,
                'instance'   => $license['instance'],
                'last_check' => time(),
            ]);
            return;
        }

        // success === true → স্বাভাবিক valid আপডেট
        $expires_from_api = $this->safe_expires($license_data);
        update_option($this->license_option, [
            'key'        => $license['key'],
            'status'     => 'valid',
            'expires'    => $expires_from_api,
            'instance'   => $license['instance'],
            'last_check' => time(),
        ]);
    }

    /** ===== Helpers for precise time & day math ===== */

    private function wp_tz(): DateTimeZone {
        return function_exists('wp_timezone')
            ? wp_timezone()
            : new DateTimeZone( get_option('timezone_string') ?: 'UTC' );
    }

    /** Parse API expiry as-is. Return 'lifetime' string or DateTimeImmutable */
    private function parse_expiry($expires_raw, DateTimeZone $tz) {
        if (!$expires_raw) return null;
        $v = strtolower(trim($expires_raw));
        if ($v === 'lifetime' || $v === 'unlimited') return 'lifetime';
        try {
            return new DateTimeImmutable($expires_raw, $tz);
        } catch (\Exception $e) {
            return null;
        }
    }

    /** Inclusive remaining days using exact seconds. Positive => future, Negative => past */
    private function remaining_days_int(DateTimeImmutable $expiry, DateTimeImmutable $now): int {
        $delta = $expiry->getTimestamp() - $now->getTimestamp();
        if ($delta === 0)  return 0;
        if ($delta > 0)    return (int) ceil($delta / DAY_IN_SECONDS);
        return - (int) ceil(abs($delta) / DAY_IN_SECONDS);
    }

    /** ===== Admin Page ===== */
    public function license_page() {
        $license         = get_option($this->license_option);
        $license_key     = isset($license['key'])    ? $license['key']    : '';
        $license_status  = isset($license['status']) ? $license['status'] : 'invalid';
        $license_expires = isset($license['expires'])? $license['expires']: null;
        ?>
        <div class="wrap">
            <h1 class="ogp-title"><?php esc_html_e('License','order-growth-plugin'); ?></h1>

            <hr class="align-center">
            <style>.ogp-title{font-size:24px;color:#333;font-weight:700;margin:0 0 6px;}</style>

            <?php settings_errors('ogp_ic_license'); ?>

            <form method="post" action="">
                <?php wp_nonce_field('ogp_ic_license_nonce', 'ogp_ic_license_nonce'); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('License Key','order-growth-plugin'); ?></th>
                        <td>
                            <input type="text"
                                   name="ogp_ic_license_key"
                                   class="regular-text"
                                   value="<?php echo esc_attr($license_key); ?>"
                                   <?php echo $license_status === 'valid' ? 'disabled' : ''; ?>>

                            <?php if ($license_status === 'valid'): ?>
                                <span style="color:green;margin-left:10px;">✓ <?php esc_html_e('Active','order-growth-plugin'); ?></span>
                            <?php endif; ?>
                        </td>
                    </tr>

                <?php if ( $license_status === 'valid' && $license_expires ) : ?>
                    <?php
                    $tz      = $this->wp_tz();
                    $now     = new DateTimeImmutable('now', $tz);
                    $expiry  = $this->parse_expiry($license_expires, $tz);
                    $is_life = ($expiry === 'lifetime');

                    $days_left_text = '';
                    if (!$is_life && $expiry instanceof DateTimeImmutable) {
                        $d = $this->remaining_days_int($expiry, $now);
                        if ($d < 0) $d = 0;
                        $days_left_text = sprintf(
                            _n('Expiring in %d day','Expiring in %d days', $d, 'order-growth-plugin'),
                            $d
                        );
                    }
                    ?>
                    <tr>
                        <th scope="row"><?php esc_html_e('License Status','order-growth-plugin'); ?></th>
                        <td>
                            <span style="display:inline-flex;align-items:center;gap:4px;padding:3px 11px;border-radius:6px;background:#ffffff;border:2px solid #10b981;color:#065f46;font-weight:700;letter-spacing:.2px;text-transform:uppercase;">
                                ✔ <?php esc_html_e('Active','order-growth-plugin'); ?>
                            </span>
                            <span style="margin-left:10px;font-weight:800;color:#065f46;">(
                                <?php
                                if ( $is_life ) {
                                    esc_html_e('Lifetime','order-growth-plugin');
                                } elseif ( $expiry instanceof DateTimeImmutable ) {
                                    echo esc_html($days_left_text);
                                }
                                ?>
                                )
                            </span>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('License Expires','order-growth-plugin'); ?></th>
                        <td>
                            <?php
                            if ($is_life) {
                                echo esc_html__('Lifetime','order-growth-plugin');
                            } elseif ($expiry instanceof DateTimeImmutable) {
                                echo esc_html($expiry->format('F j, Y g:i A'));
                                $d = $this->remaining_days_int($expiry, $now);
                                if ($d < 0) $d = 0;
                                $days_text = sprintf(
                                    _n('Expires in %d day','Expires in %d days', $d, 'order-growth-plugin'),
                                    $d
                                );
                                echo ' <span style="color:#92400e;font-weight:700;">(' . esc_html($days_text) . ')</span>';
                            } else {
                                esc_html_e('Unknown','order-growth-plugin');
                            }
                            ?>
                        </td>
                    </tr>
                <?php endif; ?>
                </table>

                <?php if ($license_status !== 'valid'): ?>
                    <p class="submit">
                        <input type="submit" name="ogp_ic_license_activate" class="button button-primary" value="Activate License">
                    </p>

                    <div class="ic-license-notice" style="margin-top:20px;padding:16px 18px;background:linear-gradient(90deg,#ecfdf5,#ffffff);border-radius:12px;box-shadow:0 8px 22px rgba(16,185,129,.12);border:1px solid #bbf7d0;">
                        <h3 style="margin:0 0 6px;font-weight:900;font-size:18px;letter-spacing:.2px;color:#059669;">License Activation Needed</h3>
                        <p style="margin:6px 0 2px;color:#065f46;font-weight:700;">Incomplete Checkouts requires an active license. Enter your license key above and click "Activate License".</p>
                        <p style="margin:4px 0 0;">
                            <span style="display:inline-block;background:#a10e10;color:#ffffff;padding:2px 8px;border-radius:999px;font-weight:800;">Don't have a key yet?</span>
                            <a href="https://www.order-growth-plugin.inventrix-it.com"
                               target="_blank"
                               rel="noopener"
                               style="display:inline-block;margin-left:8px;padding:6px 10px;border-radius:8px;text-decoration:none;font-weight:800;background:#10b981;color:#000000;border:1px solid #86efac;box-shadow:0 2px 0 rgba(0,0,0,.06);">
                                Purchase on our website →
                            </a>
                        </p>
                    </div>
                <?php else: ?>
                    <p class="submit">
                        <input type="submit" name="ogp_ic_license_deactivate" class="button button-secondary" value="Deactivate License">
                    </p>
                <?php endif; ?>
            </form>
        </div>
        <?php
    }
}
