<?php
if (!defined('ABSPATH')) exit;

/**
 * Incomplete Checkout — Main (OGP Module)
 * - Blocks real Woo orders from being tracked as "incomplete"
 * - Tracks only true Pending leads (with non-empty products)
 * - Logs Analytics events ONLY for UI actions (Save/Delete/Cron)
 *
 * [ATTEMPT-ID] per-attempt token model + rotation/TTL.
 * [PHONE MERGE] Same phone => always update the same Pending row (no new rows),
 *               even if attempt_id changes (back/refresh).
 * [SAFE CLEAR]  Checkout লোডে একবার অটোফিল সাফ; টাইপ শুরু হলে আর সাফ নয়;
 *               POST/DOING_AJAX হলে server-side কখনো খালি করবে না।
 */

if (!defined('OGP_IC_VERSION')) define('OGP_IC_VERSION', '2.0');
if (!defined('OGP_IC_PLUGIN_FILE')) define('OGP_IC_PLUGIN_FILE', __FILE__);
if (!defined('OGP_IC_PLUGIN_DIR')) define('OGP_IC_PLUGIN_DIR', plugin_dir_path(__FILE__));
if (!defined('OGP_IC_PLUGIN_URL')) define('OGP_IC_PLUGIN_URL', plugin_dir_url(__FILE__));

// === Includes ===
require_once OGP_IC_PLUGIN_DIR . 'includes/ogp_ic_license.php';
require_once OGP_IC_PLUGIN_DIR . 'includes/ogp_ic_analytics_graph.php';

global $ogp_ic_license_handler;
if (!isset($ogp_ic_license_handler) || !($ogp_ic_license_handler instanceof OGP_IC_License_Handler)) {
    $ogp_ic_license_handler = new OGP_IC_License_Handler();
}
$license_handler = $ogp_ic_license_handler;

/** Create (or update) incomplete checkout table */
add_action('admin_init', 'ogp_ic_create_incomplete_checkout_table');
function ogp_ic_create_incomplete_checkout_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        date datetime NOT NULL,
        name varchar(255) DEFAULT '',
        phone varchar(20) NOT NULL,
        address text DEFAULT '',
        ip_address varchar(45) DEFAULT '',
        subtotal decimal(10,2) DEFAULT 0.00,
        products longtext DEFAULT '',
        status enum('Pending','Completed','Cancelled','Auto Cleanup') NOT NULL DEFAULT 'Pending',
        note text DEFAULT '',
        created_at datetime NOT NULL,
        updated_at datetime NOT NULL,
        session_id varchar(100) DEFAULT '',
        attempt_id varchar(64) NOT NULL DEFAULT '',
        linked_order_id bigint(20) UNSIGNED NULL DEFAULT 0,
        PRIMARY KEY (id),
        KEY phone (phone),
        KEY status (status),
        KEY created_at (created_at),
        KEY session_id (session_id),
        UNIQUE KEY attempt_id (attempt_id),
        KEY linked_order_id (linked_order_id)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);

    // safety migrations
    $cols = $wpdb->get_col("SHOW COLUMNS FROM $table_name LIKE 'attempt_id'");
    if (!$cols) {
        $wpdb->query("ALTER TABLE $table_name ADD attempt_id varchar(64) NOT NULL DEFAULT ''");
        $wpdb->query("ALTER TABLE $table_name ADD UNIQUE KEY attempt_id (attempt_id)");
    }
    $cols = $wpdb->get_col("SHOW COLUMNS FROM $table_name LIKE 'linked_order_id'");
    if (!$cols) {
        $wpdb->query("ALTER TABLE $table_name ADD linked_order_id bigint(20) UNSIGNED NULL DEFAULT 0");
        $wpdb->query("ALTER TABLE $table_name ADD KEY linked_order_id (linked_order_id)");
    }
}

// Only run full features when license is valid
if ($license_handler->is_license_valid()) {

    // Admin UI
    add_action('plugins_loaded', function () {
        require_once OGP_IC_PLUGIN_DIR . 'includes/ogp_ic_admin.php';
    });

    // === Frontend tracker (Attempt-ID with rotation/TTL) + SAFE CLEAR ===
    add_action('woocommerce_after_checkout_form', function () {
        ?>
        <script type="text/javascript">
        jQuery(function ($) {
            let lastData = {};
            let sessionId = ogp_ic_getCookie('ogp_ic_abandoned_session_id');

            /* ==== SAFE CLEAR: একবার অটোফিল সাফ; ইউজার টাইপ করলেই আর নয় ==== */
            (function safeClearOnCheckout(){
              const $fields = jQuery('#billing_first_name,#billing_last_name,#billing_phone,#billing_address_1,#billing_city,#billing_state,#billing_postcode,#billing_email');

              let userInteracted = false;
              let clearedOnce    = false;

              function markInteracted(){ userInteracted = true; }
              $fields.on('input change focus', markInteracted);

              function clearIfNeeded(){
                if (clearedOnce || userInteracted) return;
                const hasAnyPrefill = $fields.filter(function(){
                  const v = jQuery(this).val();
                  return v && String(v).trim() !== '';
                }).length > 0;
                if (hasAnyPrefill){
                  $fields.val('');
                  clearedOnce = true;
                }
              }

              setTimeout(clearIfNeeded, 200);
              setTimeout(clearIfNeeded, 800);

              const expireAt = Date.now() + 2000;
              jQuery(document.body).on('updated_checkout', function(){
                if (Date.now() <= expireAt) clearIfNeeded();
              });

              jQuery('form.checkout').attr('autocomplete','off');
            })();
            /* ==== /SAFE CLEAR ==== */

            const TTL_MINUTES = 60;
            function newUUID(){try{return(crypto&&crypto.randomUUID)?crypto.randomUUID():(Date.now()+'-'+Math.random())}catch(e){return(Date.now()+'-'+Math.random())}}
            function loadAttemptId(){
                try{
                    const now=Date.now();
                    let a=sessionStorage.getItem('ogp_ic_attempt_id');
                    let exp=parseInt(sessionStorage.getItem('ogp_ic_attempt_expires_at')||'0',10);
                    if(!a||!exp||now>exp){
                        a=newUUID();
                        sessionStorage.setItem('ogp_ic_attempt_id',a);
                        sessionStorage.setItem('ogp_ic_attempt_expires_at', String(now+TTL_MINUTES*60*1000));
                    }
                    return a;
                }catch(e){return newUUID()}
            }
            let attemptId = loadAttemptId();
            function rotateAttemptId(){
                try{
                    const now=Date.now();
                    attemptId=newUUID();
                    sessionStorage.setItem('ogp_ic_attempt_id',attemptId);
                    sessionStorage.setItem('ogp_ic_attempt_expires_at', String(now+TTL_MINUTES*60*1000));
                    ensureAttemptInput();
                }catch(e){}
            }

            function ensureAttemptInput(){
                const $form=jQuery('form.checkout'); if(!$form.length) return;
                let $inp=$form.find('input[name="ogp_ic_attempt_id"]');
                if(!$inp.length){ $inp=jQuery('<input type="hidden" name="ogp_ic_attempt_id"/>'); $form.append($inp); }
                $inp.val(attemptId);
            }
            ensureAttemptInput();

            function ogp_ic_getCookie(name){const v=`; ${document.cookie}`; const p=v.split(`; ${name}=`); if(p.length===2) return p.pop().split(';').shift(); return '';}
            function ogp_ic_isValidPhone(ph){ return /^01[0-9]{9}$/.test(ph); }

            function ogp_ic_getData(){
                return {
                    phone: $('#billing_phone').val(),
                    first_name: $('#billing_first_name').val(),
                    last_name: $('#billing_last_name').val(),
                    address: $('#billing_address_1').val(),
                    state: $('#billing_state').val(),
                    session_id: sessionId,
                    attempt_id: attemptId
                };
            }

            /* ==== FOREVER PAUSE (২০স গ্রেস + pagehide ইমিডিয়েট) ==== */
            (function(){
              var KEY = 'ogp_ic_forever_pause';
              var pauseTimer = null;

              window.ogp_ic_pauseTabForever = function(){ try{ sessionStorage.setItem(KEY,'1'); }catch(e){} };
              window.ogp_ic_isTabForeverPaused = function(){ try{ return sessionStorage.getItem(KEY)==='1'; }catch(e){ return false; } };

              // hidden হলে ২০সেক পরে pause; আগেই ফিরে আসলে cancel
              document.addEventListener('visibilitychange', function(){
                if (document.hidden) {
              pauseTimer = setTimeout(function(){ window.ogp_ic_pauseTabForever(); }, 60000); // 60s
                } else if (pauseTimer) {
                  clearTimeout(pauseTimer); pauseTimer = null;
                }
              }, {capture:true});

              // পেজ ছেড়ে গেলে (navigate/back/bfcache) সাথে সাথে pause
              window.addEventListener('pagehide', window.ogp_ic_pauseTabForever, {capture:true});

              // ⚠ blur-এ pause দেব না—অনেক প্লাগিন blur ফায়ার করে
              // window.addEventListener('blur', window.ogp_ic_pauseTabForever);
            })();
            /* ==== /FOREVER PAUSE ==== */

            function ogp_ic_sendData(d){
                // ⛔ এই ট্যাবে autosave OFF হলে কোনো সেভই হবে না
                if (window.ogp_ic_isTabForeverPaused && window.ogp_ic_isTabForeverPaused()) return;

                ensureAttemptInput();
                $.post('<?php echo admin_url('admin-ajax.php'); ?>', {
                    action:'ogp_ic_save_abandoned_lead',
                    nonce:'<?php echo wp_create_nonce('ogp_ic_abandon_nonce'); ?>',
                    ...d
                }).done(function(res){
                    if(res && res.data){
                        if(res.data.session_id){
                            sessionId = res.data.session_id;
                            document.cookie = "ogp_ic_abandoned_session_id="+sessionId+"; path=/; max-age=86400";
                        }
                        if(res.data.rotate_attempt===true){
                            rotateAttemptId();
                            const nd=ogp_ic_getData();
                            if(nd.phone && ogp_ic_isValidPhone(nd.phone)){
                                $.post('<?php echo admin_url('admin-ajax.php'); ?>', {
                                    action:'ogp_ic_save_abandoned_lead',
                                    nonce:'<?php echo wp_create_nonce('ogp_ic_abandon_nonce'); ?>',
                                    ...nd
                                });
                            }
                        }
                    }
                });
            }

            // === Heartbeat (10s) — শুধু visible + not-paused হলে
            setInterval(function(){
                if ((window.ogp_ic_isTabForeverPaused && window.ogp_ic_isTabForeverPaused()) || document.hidden) return;
                const d=ogp_ic_getData();
                if(d.phone && ogp_ic_isValidPhone(d.phone)) ogp_ic_sendData(d);
            },10000);

            // input debounce — 0.8s পরই ইমিডিয়েট POST (phone valid + not paused + visible)
            let t;
            $('#billing_phone,#billing_first_name,#billing_last_name,#billing_address_1,#billing_state')
              .on('change keyup', function(){
                clearTimeout(t);
                t=setTimeout(function(){
                    const d=ogp_ic_getData();
                    if(!ogp_ic_isValidPhone(d.phone)) return;
                    if(JSON.stringify(d)===JSON.stringify(lastData)) return;
                    lastData=d;

                    if (!(window.ogp_ic_isTabForeverPaused && window.ogp_ic_isTabForeverPaused()) && !document.hidden) {
                        ogp_ic_sendData(d);
                    }
                },800);
              });

            // rotate once when phone first becomes valid
            let phoneWasValid=false;
            $('#billing_phone').on('keyup change', function(){
                const ok=ogp_ic_isValidPhone($('#billing_phone').val());
                if(ok && !phoneWasValid){ phoneWasValid=true; rotateAttemptId(); }
                if(!ok) phoneWasValid=false;
            });

            // rotate once on updated_checkout (cart refresh)
            let rotatedOnUpdated=false;
            jQuery(document.body).on('updated_checkout', function(){
                if(!rotatedOnUpdated){ rotatedOnUpdated=true; rotateAttemptId(); }
            });

            // ❌ hidden হলে জোর করে sendData করা হচ্ছিল—এটা সরিয়ে দেওয়া হলো (pause লজিক নষ্ট করত)
            // document.addEventListener('visibilitychange', function(){
            //     if(document.visibilityState==='hidden'){
            //         const d=ogp_ic_getData();
            //         if(d.phone && ogp_ic_isValidPhone(d.phone)) ogp_ic_sendData(d);
            //     }
            // });

            // beforeunload — কেবল not-paused অবস্থায়ই বীকন পাঠাব
            $(window).on('beforeunload', function(){
                if (window.ogp_ic_isTabForeverPaused && window.ogp_ic_isTabForeverPaused()) return;
                const d=ogp_ic_getData();
                if(d.phone && ogp_ic_isValidPhone(d.phone)){
                    if(navigator.sendBeacon){
                        const fd=new FormData();
                        fd.append('action','ogp_ic_save_abandoned_lead');
                        fd.append('nonce','<?php echo wp_create_nonce('ogp_ic_abandon_nonce'); ?>');
                        Object.keys(d).forEach(k=>fd.append(k,d[k]));
                        navigator.sendBeacon('<?php echo admin_url('admin-ajax.php'); ?>', fd);
                    } else {
                        ogp_ic_sendData(d);
                    }
                }
            });
        });
        </script>
        <?php
    });

    /* === ALWAYS EMPTY DEFAULTS ON CHECKOUT (server-side, but never during POST/AJAX) === */
    function ogp_ic_return_empty_string(){ return ''; }

    add_filter('woocommerce_checkout_get_value','ogp_ic_force_empty_checkout_values', 9999, 2);
    function ogp_ic_force_empty_checkout_values($value, $input_key){
        if ($_SERVER['REQUEST_METHOD'] === 'POST' || (defined('DOING_AJAX') && DOING_AJAX)) {
            return $value;
        }
        $keys = [
            'billing_first_name','billing_last_name','billing_phone',
            'billing_address_1','billing_city','billing_state',
            'billing_postcode','billing_email'
        ];
        if (in_array($input_key, $keys, true)) {
            return '';
        }
        return $value;
    }

    add_filter('default_checkout_billing_first_name','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_last_name','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_phone','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_address_1','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_city','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_state','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_postcode','ogp_ic_return_empty_string');
    add_filter('default_checkout_billing_email','ogp_ic_return_empty_string');

    add_filter('woocommerce_checkout_fields','ogp_ic_disable_autofill_globally', 9999);
    function ogp_ic_disable_autofill_globally($fields){
        $targets = [
            'billing_first_name','billing_last_name','billing_phone',
            'billing_address_1','billing_city','billing_state',
            'billing_postcode','billing_email'
        ];
        foreach ($targets as $k){
            if (isset($fields['billing'][$k])) {
                $fields['billing'][$k]['default'] = '';
                $fields['billing'][$k]['custom_attributes'] = array_merge(
                    $fields['billing'][$k]['custom_attributes'] ?? [],
                    [
                        'autocomplete'   => 'off',
                        'autocapitalize' => 'none',
                        'autocorrect'    => 'off',
                        'spellcheck'     => 'false',
                        'inputmode'      => 'text',
                    ]
                );
            }
        }
        return $fields;
    }
    /* === /SAFE CLEAR === */

    add_action('wp_ajax_nopriv_ogp_ic_save_abandoned_lead', 'ogp_ic_save_abandoned_lead');
    add_action('wp_ajax_ogp_ic_save_abandoned_lead',      'ogp_ic_save_abandoned_lead');

    function ogp_ic_save_abandoned_lead() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_abandon_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        $phone = sanitize_text_field($_POST['phone'] ?? '');
        if (!preg_match('/^01[0-9]{9}$/', $phone)) {
            wp_send_json_error('Invalid phone format');
        }

        $first = sanitize_text_field($_POST['first_name'] ?? '');
        $last  = sanitize_text_field($_POST['last_name'] ?? '');
        if (in_array(strtolower($first), ['undefined','null'], true)) $first = '';
        if (in_array(strtolower($last),  ['undefined','null'],  true)) $last  = '';
        $name = trim("$first $last");
        $addr = sanitize_text_field($_POST['address'] ?? '');

        $ip_address = '';
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip_address = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip_address = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } else {
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? '';
        }
        $ip_address = filter_var($ip_address, FILTER_VALIDATE_IP) ?: '';

        $cart = WC()->cart ? WC()->cart->get_cart() : [];
        $products = [];
        foreach ($cart as $item) {
            if (!empty($item['data']) && method_exists($item['data'],'get_name')) {
                $products[] = $item['data']->get_name() . ' x' . intval($item['quantity']);
            }
        }
        $product_str = implode(', ', $products);
        $subtotal    = WC()->cart ? WC()->cart->get_subtotal() : 0.00;

        $current_time = current_time('mysql');
        $session_id   = sanitize_text_field($_POST['session_id'] ?? '');
        if (empty($session_id)) $session_id = md5($phone . $ip_address . microtime(true));

        $attempt_id = sanitize_text_field($_POST['attempt_id'] ?? '');

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';

        if (trim($product_str) === '') {
            wp_send_json_success([
                'message'    => 'Skipped: empty products',
                'skipped'    => true,
                'session_id' => $session_id,
            ]);
        }

        if ($attempt_id !== '') {
            $orders = wc_get_orders([
                'limit'      => 1,
                'meta_key'   => '_ogp_ic_attempt_id',
                'meta_value' => $attempt_id,
                'return'     => 'ids',
            ]);
            if (!empty($orders)) {
                $wpdb->query( $wpdb->prepare(
                    "DELETE FROM {$table_name} WHERE attempt_id = %s AND status = 'Pending'",
                    $attempt_id
                ));
                wp_send_json_success([
                    'message'        => 'Skipped: attempt is already bound to a Woo order',
                    'skipped'        => true,
                    'rotate_attempt' => true,
                    'session_id'     => $session_id,
                ]);
            }
        }

        $existing_by_attempt = null;
        if ($attempt_id !== '') {
            $existing_by_attempt = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_name WHERE attempt_id = %s AND status = 'Pending' LIMIT 1",
                $attempt_id
            ));
        }
        $existing_by_phone = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE phone = %s AND status = 'Pending' ORDER BY created_at DESC LIMIT 1",
            $phone
        ));

        $target = $existing_by_attempt ?: $existing_by_phone;

        if ($target) {
            $update = [
                'name'       => $name,
                'address'    => $addr,
                'ip_address' => $ip_address,
                'subtotal'   => $subtotal,
                'products'   => $product_str,
                'updated_at' => $current_time,
                'session_id' => $session_id,
            ];
            if (!$existing_by_attempt && $attempt_id !== '') {
                $update['attempt_id'] = $attempt_id;
            }

            $formats = ['%s','%s','%s','%f','%s','%s','%s'];
            if (isset($update['attempt_id'])) { $formats[] = '%s'; }

            $wpdb->update(
                $table_name,
                $update,
                ['id' => $target->id],
                $formats,
                ['%d']
            );

            $wpdb->query( $wpdb->prepare(
                "DELETE FROM {$table_name} WHERE phone = %s AND status = 'Pending' AND id <> %d",
                $phone, $target->id
            ));

            wp_send_json_success([
                'message'    => 'Updated existing lead (merged by phone)',
                'session_id' => $session_id
            ]);
        } else {
            $wpdb->insert(
                $table_name,
                [
                    'date'           => $current_time,
                    'name'           => $name,
                    'phone'          => $phone,
                    'address'        => $addr,
                    'ip_address'     => $ip_address,
                    'subtotal'       => $subtotal,
                    'products'       => $product_str,
                    'status'         => 'Pending',
                    'note'           => '',
                    'created_at'     => $current_time,
                    'updated_at'     => $current_time,
                    'session_id'     => $session_id,
                    'attempt_id'     => $attempt_id,
                    'linked_order_id'=> 0,
                ],
                ['%s','%s','%s','%s','%s','%f','%s','%s','%s','%s','%s','%s','%s','%d']
            );

            wp_send_json_success([
                'message'    => 'Created new lead',
                'session_id' => $session_id
            ]);
        }
    }

    add_action('woocommerce_checkout_create_order', function($order, $data){
        $attempt_id = isset($_POST['ogp_ic_attempt_id']) ? sanitize_text_field($_POST['ogp_ic_attempt_id']) : '';
        if ($attempt_id) {
            $order->update_meta_data('_ogp_ic_attempt_id', $attempt_id);
        }
    }, 10, 2);

    add_action('woocommerce_checkout_order_processed', 'ogp_ic_process_recovered_leads');
    function ogp_ic_process_recovered_leads($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) return;

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';

        $attempt = (string) $order->get_meta('_ogp_ic_attempt_id');
        $phone   = preg_replace('/\D/', '', (string)$order->get_billing_phone());

        if (!empty($attempt)) {
            $wpdb->query( $wpdb->prepare(
                "DELETE FROM {$table_name} WHERE attempt_id = %s AND status = 'Pending'",
                $attempt
            ));
        }
        if (!empty($phone)) {
            $wpdb->query( $wpdb->prepare(
                "DELETE FROM {$table_name}
                 WHERE phone = %s AND status = 'Pending'
                 AND created_at >= DATE_SUB(NOW(), INTERVAL 1 DAY)",
                $phone
            ));
        }

        if (!headers_sent()) {
            setcookie('ogp_ic_abandoned_session_id', '', time() - 3600, '/');
        }
    }

    add_action('woocommerce_thankyou', function($order_id){
        ?>
        <script>
        try{
            sessionStorage.removeItem('ogp_ic_attempt_id');
            sessionStorage.removeItem('ogp_ic_attempt_expires_at');
            localStorage.removeItem('ogp_ic_attempt_id');
            localStorage.removeItem('ogp_ic_attempt_expires_at');
        }catch(e){}
        </script>
        <?php
    }, 999);

    add_action('wp_ajax_ogp_ic_update_abandoned_note', function () {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_abandon_note_nonce')) {
            wp_send_json_error(['reason' => 'Invalid nonce']);
        }
        $id   = intval($_POST['lead_id'] ?? 0);
        $note = sanitize_textarea_field($_POST['note'] ?? '');
        if (!$id) wp_send_json_error(['reason' => 'Invalid ID']);

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';
        $ok = $wpdb->update(
            $table_name,
            ['note' => $note, 'updated_at' => current_time('mysql')],
            ['id' => $id],
            ['%s','%s'],
            ['%d']
        );
        $ok !== false ? wp_send_json_success(['updated' => true])
                      : wp_send_json_error(['reason' => 'Failed to update note']);
    });

    add_action('wp_ajax_ogp_ic_delete_abandoned_lead', function () {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_abandon_note_nonce')) {
            wp_send_json_error(['reason' => 'Invalid nonce']);
        }
        $id = intval($_POST['lead_id'] ?? 0);
        if (!$id) wp_send_json_error(['reason' => 'Invalid ID']);

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';
        $lead = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $id));
        if (!$lead) wp_send_json_error(['reason' => 'Lead not found']);

        ogp_ic_log_event($lead->id, $lead->phone, 'cancelled', $lead->subtotal, $lead->ip_address);

        $deleted = $wpdb->delete($table_name, ['id' => $id], ['%d']);
        $deleted ? wp_send_json_success(['deleted' => true])
                 : wp_send_json_error(['reason' => 'Failed to delete lead']);
    });

    add_action('wp_ajax_ogp_ic_bulk_delete_abandoned_leads', function () {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_abandon_note_nonce')) {
            wp_send_json_error(['reason' => 'Invalid nonce']);
        }
        $lead_ids = isset($_POST['lead_ids']) && is_array($_POST['lead_ids']) ? array_map('intval', $_POST['lead_ids']) : [];
        if (!$lead_ids) wp_send_json_error(['reason' => 'No leads selected']);

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';
        $deleted_count = 0;

        foreach ($lead_ids as $id) {
            $lead = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $id));
            if ($lead) {
                ogp_ic_log_event($lead->id, $lead->phone, 'cancelled', $lead->subtotal, $lead->ip_address);
                if ($wpdb->delete($table_name, ['id' => $id], ['%d'])) $deleted_count++;
            }
        }
        wp_send_json_success(['deleted' => $deleted_count]);
    });

    add_action('wp_ajax_ogp_ic_export_selected_abandoned_checkouts', function () {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_abandon_note_nonce')) {
            wp_die('Security check failed');
        }
        $lead_ids = isset($_POST['lead_ids']) && is_array($_POST['lead_ids']) ? array_map('intval', $_POST['lead_ids']) : [];
        if (!$lead_ids) wp_die('No leads selected');

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="selected-incomplete-checkouts.csv' . '"');

        $out = fopen('php://output', 'w');
        fputcsv($out, ['Date','Name','Phone','Address','IP Address','Subtotal','Products','Status','Note']);

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';
        foreach ($lead_ids as $id) {
            $lead = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $id));
            if ($lead) {
                fputcsv($out, [
                    $lead->date, $lead->name, $lead->phone, $lead->address,
                    $lead->ip_address, $lead->subtotal, $lead->products,
                    $lead->status, $lead->note
                ]);
            }
        }
        fclose($out);
        exit;
    });

    add_action('wp_ajax_ogp_ic_convert_abandoned_to_order', function () {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_abandon_note_nonce')) {
            wp_send_json_error(['reason' => 'Security check failed - invalid nonce']);
        }
        if ( ! current_user_can('edit_shop_orders') && ! current_user_can('manage_woocommerce') ) {
            wp_send_json_error(['reason' => 'Permission denied - insufficient privileges']);
        }

        $lead_id = intval($_POST['lead_id'] ?? 0);
        if (!$lead_id) wp_send_json_error(['reason' => 'Invalid lead ID']);

        global $wpdb;
        $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';
        $lead = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $lead_id));
        if (!$lead) wp_send_json_error(['reason' => 'Lead not found']);

        if (!class_exists('WooCommerce') || !function_exists('wc_create_order')) {
            wp_send_json_error(['reason' => 'WooCommerce is not active or available']);
        }

        try {
            $name_parts = explode(' ', (string)$lead->name, 2);
            $first_name = $name_parts[0] ?? '';
            $last_name  = $name_parts[1] ?? '';

            if (empty($lead->phone))    wp_send_json_error(['reason' => 'Missing required phone number']);
            if (empty($lead->products)) wp_send_json_error(['reason' => 'No products found in abandoned checkout']);

            $order = wc_create_order();
            if (!$order) wp_send_json_error(['reason' => 'Failed to create WooCommerce order']);

            $order->set_billing_first_name($first_name);
            $order->set_billing_last_name($last_name);
            $order->set_billing_phone($lead->phone);
            $order->set_billing_address_1($lead->address);
            $order->set_billing_country('BD');

            $add_product_by_name_or_sku = function($needle, $qty) {
                if (function_exists('wc_get_product_id_by_sku')) {
                    $pid = wc_get_product_id_by_sku($needle);
                    if ($pid) {
                        $p = wc_get_product($pid);
                        if ($p) return [$p, $qty];
                    }
                }
                $posts = get_posts([
                    'post_type'   => 'product',
                    's'           => $needle,
                    'post_status' => 'publish',
                    'numberposts' => 1,
                ]);
                if ($posts) {
                    $p = wc_get_product($posts[0]->ID);
                    if ($p) return [$p, $qty];
                }
                return [null, 0];
            };

            $products_added = 0;
            foreach (explode(', ', $lead->products) as $line) {
                if (preg_match('/^(.+?)\s+x(\d+)$/', trim($line), $m)) {
                    $needle = trim($m[1]); $qty = max(1, intval($m[2]));
                    [$product, $q] = $add_product_by_name_or_sku($needle, $qty);
                    if ($product) { $order->add_product($product, $q); $products_added++; }
                }
            }
            if ($products_added === 0) {
                $item = new WC_Order_Item_Product();
                $item->set_name('Recovered Order: ' . $lead->products);
                $item->set_quantity(1);
                $item->set_subtotal((float)$lead->subtotal);
                $item->set_total((float)$lead->subtotal);
                $order->add_item($item);
            }

            $order->set_payment_method('cod');
            $order->set_payment_method_title('Cash on delivery');
            $order->calculate_totals();
            $order->set_status('processing');
            $order->add_order_note('Converted from Incomplete Checkout #' . $lead_id . '.', 0, true);

            if (!empty($lead->attempt_id)) {
                $order->update_meta_data('_ogp_ic_attempt_id', $lead->attempt_id);
            }

            $order->save();
            $order_id = $order->get_id();
            if (!$order_id) wp_send_json_error(['reason' => 'Failed to save WooCommerce order']);

            ogp_ic_log_event($lead->id, $lead->phone, 'completed', $lead->subtotal, $lead->ip_address);

            $wpdb->delete($table_name, ['id' => $lead_id], ['%d']);

            error_log("OGP convert: Lead {$lead_id} -> Order {$order_id}");
            wp_send_json_success(['order_id' => $order_id, 'message' => 'Successfully converted to Order #'.$order_id]);

        } catch (Throwable $e) {
            error_log('OGP convert error (lead '.$lead_id.'): '.$e->getMessage());
            wp_send_json_error(['reason' => 'Conversion failed: '.$e->getMessage()]);
        }
    });

} else {
    // License validation message will be shown within OGP's menu structure instead
}

register_activation_hook(OGP_IC_PLUGIN_FILE, 'ogp_ic_plugin_activated');
function ogp_ic_plugin_activated() { ogp_ic_update_cleanup_schedule(); }

register_deactivation_hook(OGP_IC_PLUGIN_FILE, function() {
    global $wpdb;
    foreach ([$wpdb->prefix.'ogp_incomplete_checkout', $wpdb->prefix.'ogp_analytics'] as $t) {
        $wpdb->query("DROP TABLE IF EXISTS {$t}");
    }
    delete_option('ogp_ic_abandoned_cleanup_interval');
    delete_option('ogp_ic_last_cleanup_time');
    delete_option('ogp_ic_cleanup_schedule_updated');
    wp_clear_scheduled_hook('ogp_ic_cleanup_old_abandoned_leads_event');
    if (class_exists('OGP_IC_License_Handler')) {
        OGP_IC_License_Handler::deactivate();
    }
});

add_filter('cron_schedules', 'ogp_ic_add_cron_schedules');
function ogp_ic_add_cron_schedules($s) {
    if (!isset($s['daily'])) {
        $s['daily'] = ['interval'=>86400, 'display'=>__('Once Daily')];
    }
    return $s;
}

add_action('ogp_ic_cleanup_old_abandoned_leads_event', 'ogp_ic_cleanup_old_abandoned_leads');

function ogp_ic_update_cleanup_schedule() {
    wp_clear_scheduled_hook('ogp_ic_cleanup_old_abandoned_leads_event');
    if (!wp_next_scheduled('ogp_ic_cleanup_old_abandoned_leads_event')) {
        wp_schedule_event(time(), 'daily', 'ogp_ic_cleanup_old_abandoned_leads_event');
    }
    update_option('ogp_ic_cleanup_schedule_updated', time());
}

function ogp_ic_cleanup_old_abandoned_leads() {
    $cleanup_interval   = get_option('ogp_ic_abandoned_cleanup_interval', 7);
    $interval_in_seconds= $cleanup_interval * 86400;
    $cutoff_time        = date_i18n('Y-m-d H:i:s', current_time('timestamp') - $interval_in_seconds, false);

    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';

    $old_leads = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $table_name WHERE status = 'Pending' AND created_at < %s",
        $cutoff_time
    ));

    $count = 0;
    foreach ($old_leads as $lead) {
        ogp_ic_log_event($lead->id, $lead->phone, 'auto_cleanup', $lead->subtotal, $lead->ip_address);
        $wpdb->delete($table_name, ['id' => $lead->id], ['%d']);
        $count++;
    }
    update_option('ogp_ic_last_cleanup_time', time());
}

add_action('wp_ajax_ogp_ic_schedule_cleanup_now', function() {
    if (!current_user_can('manage_options')) wp_send_json_error(['message'=>'Permission denied']);
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_schedule_cleanup_nonce')) {
        wp_send_json_error(['message'=>'Security check failed']);
    }
    ogp_ic_update_cleanup_schedule();
    wp_send_json_success(['message'=>'Cleanup scheduled successfully']);
});

add_action('wp_ajax_ogp_ic_run_cleanup_now', function() {
    if (!current_user_can('manage_options')) wp_send_json_error(['message'=>'Permission denied']);
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ogp_ic_run_cleanup_nonce')) {
        wp_send_json_error(['message'=>'Security check failed']);
    }

    $cleanup_interval   = get_option('ogp_ic_abandoned_cleanup_interval', 7);
    $interval_in_seconds= $cleanup_interval * 86400;
    $cutoff_time        = date('Y-m-d H:i:s', time() - $interval_in_seconds);

    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_incomplete_checkout';

    $old_leads = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $table_name WHERE status = 'Pending' AND created_at < %s",
        $cutoff_time
    ));

    $count = 0;
    foreach ($old_leads as $lead) {
        ogp_ic_log_event($lead->id, $lead->phone, 'auto_cleanup', $lead->subtotal, $lead->ip_address);
        $wpdb->delete($table_name, ['id' => $lead->id], ['%d']);
        $count++;
    }
    update_option('ogp_ic_last_cleanup_time', time());
    wp_send_json_success(['count'=>$count]);
});

function ogp_ic_convert_to_bangladesh_time($timestamp) {
    $unix = intval($timestamp) ?: time();
    $dt   = new DateTime('@'.$unix);
    $dt->setTimezone(new DateTimeZone('Asia/Dhaka'));
    return $dt->format('Y-m-d g:i:s A');
}
