<?php
if (!defined('ABSPATH')){
	exit; // Exit if accessed directly.
}

if(!defined('OGP_ABSPATH')) define('OGP_ABSPATH',dirname(__FILE__));

function ogp_get_root_url(){
    $local_path = __FILE__;
    if( DIRECTORY_SEPARATOR == '\\' ) { // Win32 fix
        $local_path = basename(dirname(__FILE__)) .'/'. basename(__FILE__);
    }
    $plugin_url = plugins_url('', $local_path);
    return $plugin_url . '/';
}

function ogp_activation_hook(){
	global $wpdb;
	$table_name = $wpdb->prefix . 'ogp_block7';
	$charset_collate = $wpdb->get_charset_collate();
	
	$sql = "CREATE TABLE IF NOT EXISTS $table_name (
    id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    type INT(10) UNSIGNED NOT NULL,
    ip VARCHAR(50) NOT NULL,
    phoneNumber VARCHAR(15) NOT NULL, 
    blockedTime BIGINT(20) UNSIGNED NOT NULL,
    reason VARCHAR(255) NOT NULL,
    lastAttempt BIGINT(20) UNSIGNED NOT NULL,
    blockedHits INT(10) UNSIGNED NOT NULL,
    expiration BIGINT(20) UNSIGNED NOT NULL,
    parameters TEXT NOT NULL,
    comment TEXT NULL
	) $charset_collate;";

	require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
	dbDelta($sql);
	
	$column_exists = $wpdb->get_results($wpdb->prepare(
		"SHOW COLUMNS FROM $table_name LIKE %s",
		'comment'
	));
	
	if (empty($column_exists)) {
		$wpdb->query("ALTER TABLE $table_name ADD COLUMN comment TEXT NULL");
	}
}

function ogp_deactivation_hook(){
    // Blocklist table (ogp_block7) is intentionally NOT dropped
    // Only incomplete checkout and analytics tables are dropped on deactivation
}
register_deactivation_hook(__FILE__, 'ogp_deactivation_hook');

function ogp_load_textdomain(){
    load_plugin_textdomain('order-growth-plugin', false, dirname(__FILE__).'/languages/');
}
add_action('plugins_loaded','ogp_load_textdomain');

function ogp_load_admin_assets($hook){
	$screen = get_current_screen();
	
	// Check if we're on a WooCommerce order list page
	$is_woo_order_page = ($screen && ($screen->id === 'edit-shop_order' || $screen->id === 'woocommerce_page_wc-orders'));
	
	// Check if we're on an OGP admin page
	$is_ogp_page = false;
	if (isset($_GET['page'])) {
		$ogp_pages = array(
			'ogp-dashboard',
			'ogp-incomplete-checkout',
			'ogp-block-list',
			'ogp-order-settings',
			'ogp-analytics',
			'ogp-settings',
			'ogp-license',
			'ogp-incomplete-checkout-analytics',
			'ogp-incomplete-checkout-settings',
			'ogp-incomplete-checkout-license'
		);
		$is_ogp_page = in_array($_GET['page'], $ogp_pages);
	}
	
	// Only load assets on OGP pages or WooCommerce order pages
	if (!$is_ogp_page && !$is_woo_order_page) {
		return;
	}
	
	wp_enqueue_style('ogp-minitoggle-style', ogp_get_root_url() .'assets/css/minitoggle.css',array(),'1.1');
	wp_enqueue_style('ogp-main-style', ogp_get_root_url() .'style.css',array(),time());
	
	wp_enqueue_script('ogp-minitoggle-js', ogp_get_root_url() .'assets/js/minitoggle.js',array('jquery'),"1.0",true);
	wp_enqueue_script('ogp-main-js', ogp_get_root_url() .'assets/js/main.js',array('jquery'),time(),true);
	wp_enqueue_script('ogp-blocklist-sync', ogp_get_root_url() .'assets/js/ogp-blocklist-sync.js', array('jquery'), time(), true);
	wp_enqueue_script('ogp-live-search', ogp_get_root_url() .'assets/js/ogp-live-search.js', array(), time(), true);
}
add_action('admin_enqueue_scripts','ogp_load_admin_assets');

function ogp_load_front_assets(){
    wp_enqueue_script('ogp-live-search', ogp_get_root_url() .'assets/js/ogp-live-search.js', array(), time(), true);
    wp_enqueue_style('ogp-main-style', ogp_get_root_url() .'style.css',array(), time());
}
add_action('wp_enqueue_scripts','ogp_load_front_assets');

function ogp_add_admin_menu() {
    add_menu_page(
        'Order Growth Plugin v1.0.2',
        'Order Growth Plugin',
        'manage_options',
        'order-growth-plugin-settings',
        'ogp_settings_page_callback',
        plugins_url() . '/order-growth-plugin/logo.png',
        6
    );
	add_submenu_page(
        'order-growth-plugin-settings',
        __('Blocking','order-growth-plugin'),
        __('Blocking','order-growth-plugin'),
        'manage_options',
        'ogp-blocking',
        'ogp_blocking_page_callback'
    );
	add_submenu_page(
        'order-growth-plugin-settings',
        __('Order Settings','order-growth-plugin'),
        __('Order Settings','order-growth-plugin'),
        'manage_options',
        'ogp-order-settings',
        'ogp_order_settings_page_callback'
    );
}
// add_action('admin_menu','ogp_add_admin_menu'); // disabled by Order Growth Plugin merge

//ogp_settings_page_callback
require_once('ogp-settings-page.php');
//ogp_blocking_page_callback
require_once('ogp-blocking-page.php');
//ogp_order_settings_page_callback
require_once('ogp-order-settings-page.php');
//ogp-handle-columns.php
require_once('ogp-handle-columns.php');

function ogp_normalize_phone_number($phone_number) {
    $normalized = preg_replace('/[^\d+]/', '', $phone_number);
    if (substr($normalized, 0, 3) === '+88') {
        $normalized = substr($normalized, 3); // Strip +88 country code
    }
    return $normalized;
}

function ogp_isMultidimensionalArray($array){
    if (!is_array($array)) {
        return false;
    }
    
    foreach ($array as $value) {
        if (is_array($value)) {
            return true;
        }
    }
    return false; 
}

function ogp_get_order_details($billing_phone){
    if(empty($billing_phone)){
        $data = [
            'message' => 'Billing Mobile no is empty.',
            'html' => 0
        ];
        return json_encode($data);
    }
    
    $billing_phone = ogp_normalize_phone_number($billing_phone);
    
    // NEW — use your proxy
    $url = 'https://order-growth-plugin-license.inventrix-it.com/courier-reports.php?phone=' . urlencode($billing_phone);

    $response = wp_remote_get($url, [
        'headers' => [
            'Accept'     => 'application/json',
            'User-Agent' => 'WordPress/Order Growth Plugin'
        ],
        'timeout'  => 30,
        'sslverify'=> true   // আপনার SSL valid হলে true রাখুন
    ]);
    if (is_wp_error($response)) {
        $response_error = $response->get_error_message();
        $data = [
            'message' => 'API Error: ' . $response_error,
            'html' => 0
        ];
        return json_encode($data);
    }

    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        $data = [
            'message' => 'API returned error code: ' . $response_code,
            'html' => 0
        ];
        return json_encode($data);
    }

    $response_body = wp_remote_retrieve_body($response);
    if (empty($response_body)) {
        $data = [
            'message' => 'No response from API.',
            'html' => 0
        ];
        return json_encode($data);
    }

    // Decode JSON response
    $response_data = json_decode($response_body, true);
    
    // Check for JSON decode errors
    if (json_last_error() !== JSON_ERROR_NONE) {
        $data = [
            'message' => 'Invalid JSON response from API: ' . json_last_error_msg(),
            'html' => 0
        ];
        return json_encode($data);
    }

    // Handle empty or null response
    if (empty($response_data)) {
        $data = [
            'message' => 'No data found for this phone number.',
            'html' => 0
        ];
        return json_encode($data);
    }

    // Check if response is multidimensional array (multiple courier data)
    $response_data_isMultidimensional = ogp_isMultidimensionalArray($response_data);

    if ($response_data_isMultidimensional === false) {
        // Single response or error message
        if (isset($response_data['message'])) {
            $data = [
                'courier' => 'no',
                'message' => $response_data['message'],
                'html' => 0
            ];
            return json_encode($data);
        } else {
            // If it's not multidimensional but has no message, treat as no data
            $data = [
                'message' => 'No courier data available for this phone number.',
                'html' => 0
            ];
            return json_encode($data);
        }
    }

    // Validate courier data structure
    $validated_data = [];
    foreach ($response_data as $courier_data) {
        if (is_array($courier_data) && isset($courier_data['courier'])) {
            $validated_data[] = $courier_data;
        }
    }

    if (empty($validated_data)) {
        $data = [
            'message' => 'Invalid courier data format received from API.',
            'html' => 0
        ];
        return json_encode($data);
    }

    return json_encode($validated_data);
}

function ogp_extractCourierData($data, $courier) {
    if (!is_array($data)) {
        return null;
    }
    
    foreach ($data as $courierData) {
        if (is_array($courierData) && isset($courierData['courier']) && $courierData['courier'] === $courier) {
            return $courierData;
        }
    }
    return null; 
}

add_action('woocommerce_admin_order_data_after_billing_address', 'ogp_display_delivery_details_after_billing_address', 10, 1);
function ogp_display_delivery_details_after_billing_address($order){
	if (!$order) {
        $data = [
			'message' => 'Order not found.',
			'html' => 0
		];
		$delivery_details = json_encode($data);
    } else {
		$billing_phone = $order->get_billing_phone();
		$delivery_details = ogp_get_order_details($billing_phone);
	}
	$delivery_details = json_decode($delivery_details, true);
?>	
	<div class="ogp-delivery-container">
		<div class="ogp-delivery-header"><?php esc_html_e('Customer Reports', 'order-growth-plugin');?></div>
		<?php 
			$delivery_details_isMultidimensional = ogp_isMultidimensionalArray($delivery_details);
			if($delivery_details_isMultidimensional === false){
				$message = isset($delivery_details['message']) ? $delivery_details['message'] : 'No data available';
				echo '<p style="font-size: 1.2rem;">'. esc_html($message) .'</p>';
			} else {
				// Initialize totals
				$successful_delivery_total = 0;
				$total_delivery_total = 0;
				$cancel_delivery_total = 0;
				
				// Initialize courier data with defaults
				$successful_delivery_pathao = 0;
				$total_delivery_pathao = 0;
				$cancel_delivery_pathao = 0;
				$success_ratio_pathao = 0;
				
				$successful_delivery_steadfast = 0;
				$total_delivery_steadfast = 0;
				$cancel_delivery_steadfast = 0;
				$success_ratio_steadfast = 0;
				
				$successful_delivery_redx = 0;
				$total_delivery_redx = 0;
				$cancel_delivery_redx = 0;
				$success_ratio_redx = 0;
				
				$successful_delivery_paperfly = 0;
				$total_delivery_paperfly = 0;
				$cancel_delivery_paperfly = 0;
				$success_ratio_paperfly = 0;
				
				// Extract courier data
				$pathaoData = ogp_extractCourierData($delivery_details, 'pathao');
				$steadfastData = ogp_extractCourierData($delivery_details, 'steadfast');
				$redxData = ogp_extractCourierData($delivery_details, 'redx');
				$paperflyData = ogp_extractCourierData($delivery_details, 'paperfly');
				
				if ($pathaoData) {
					$successful_delivery_pathao = isset($pathaoData['successful_delivery_pathao']) ? intval($pathaoData['successful_delivery_pathao']) : 0;
					$total_delivery_pathao = isset($pathaoData['total_delivery_pathao']) ? intval($pathaoData['total_delivery_pathao']) : 0;
					$cancel_delivery_pathao = $total_delivery_pathao - $successful_delivery_pathao;
					$success_ratio_pathao = ($total_delivery_pathao > 0) ? round(($successful_delivery_pathao/$total_delivery_pathao)*100) : 0;
				}

				if ($steadfastData) {
					$successful_delivery_steadfast = isset($steadfastData['successful_delivery_steadfast']) ? intval($steadfastData['successful_delivery_steadfast']) : 0;
					$cancel_delivery_steadfast = isset($steadfastData['cancel_delivery_steadfast']) ? intval($steadfastData['cancel_delivery_steadfast']) : 0;
					$total_delivery_steadfast = $successful_delivery_steadfast + $cancel_delivery_steadfast;
					$success_ratio_steadfast = ($total_delivery_steadfast > 0) ? round(($successful_delivery_steadfast/$total_delivery_steadfast)*100) : 0;
				}
				
				if ($redxData) {
					$successful_delivery_redx = isset($redxData['successful_delivery_redx']) ? intval($redxData['successful_delivery_redx']) : 0;
					$total_delivery_redx = isset($redxData['total_delivery_redx']) ? intval($redxData['total_delivery_redx']) : 0;
					$cancel_delivery_redx = $total_delivery_redx - $successful_delivery_redx;
					$success_ratio_redx = ($total_delivery_redx > 0) ? round(($successful_delivery_redx/$total_delivery_redx)*100) : 0;
				}
				
				if ($paperflyData){
					$successful_delivery_paperfly = isset($paperflyData['successful_delivery_paperfly']) ? intval($paperflyData['successful_delivery_paperfly']) : 0;
					$cancel_delivery_paperfly = isset($paperflyData['cancel_delivery_paperfly']) ? intval($paperflyData['cancel_delivery_paperfly']) : 0;
					$total_delivery_paperfly = $successful_delivery_paperfly + $cancel_delivery_paperfly;
					$success_ratio_paperfly = ($total_delivery_paperfly > 0) ? round(($successful_delivery_paperfly/$total_delivery_paperfly)*100) : 0;
				}
				
				$successful_delivery_total = $successful_delivery_pathao + $successful_delivery_steadfast + $successful_delivery_redx + $successful_delivery_paperfly;
				$cancel_delivery_total = $cancel_delivery_pathao + $cancel_delivery_steadfast + $cancel_delivery_redx + $cancel_delivery_paperfly;
				$total_delivery_total = $total_delivery_pathao + $total_delivery_steadfast + $total_delivery_redx + $total_delivery_paperfly;
				$success_ratio_total = ($total_delivery_total > 0) ? round(($successful_delivery_total/$total_delivery_total)*100) : 0;
		?>
			<div class="ogp-delivery-info">
				<div style="font-size: 1.2rem; font-weight: 600;"><?php esc_html_e('Total Parcels', 'order-growth-plugin');?></div>
				<div style="font-size: 1.2rem; font-weight: 600;"><?php echo esc_html($total_delivery_total);?></div>
			</div>
			<div class="ogp-delivery-info">
				<div style="font-size: 1.2rem; font-weight: 600;"><?php esc_html_e('Total Delivered','order-growth-plugin');?></div>
				<div style="font-size: 1.2rem; font-weight: 600;"><?php echo esc_html($successful_delivery_total);?></div>
			</div>
			<div class="ogp-delivery-info">
				<div style="font-size: 1.2rem; font-weight: 600;"><?php esc_html_e('Total Returned','order-growth-plugin');?></div>
				<div style="font-size: 1.2rem; font-weight: 600;"><?php echo esc_html($cancel_delivery_total);?></div>
			</div>
			<div class="ogp-delivery-info" style="overflow-x: scroll;">
				<table class="table table-bordered" id="" width="100%" cellspacing="0">
					<thead>
						<tr>
							<th class="left col1">Courier</th>
							<th>Total</th>
							<th>Delivered</th>
							<th>Returned</th>
							<th>Success Rate</th>
						</tr>
					</thead>
					<tbody>
						<tr>
							<td class="left col1">Pathao</td>
							<td><?php echo esc_html($total_delivery_pathao);?></td>
							<td class="bg-green"><?php echo esc_html($successful_delivery_pathao);?></td>
							<td class="bg-red"><?php echo esc_html($cancel_delivery_pathao);?></td>
							<td><?php echo esc_html($success_ratio_pathao);?>%</td>
						</tr>
						<tr>
							<td class="left col1">Steadfast</td>
							<td><?php echo esc_html($total_delivery_steadfast);?></td>
							<td class="bg-green"><?php echo esc_html($successful_delivery_steadfast);?></td>
							<td class="bg-red"><?php echo esc_html($cancel_delivery_steadfast);?></td>
							<td><?php echo esc_html($success_ratio_steadfast);?>%</td>
						</tr>
						<tr>
							<td class="left col1">RedX</td>
							<td><?php echo esc_html($total_delivery_redx);?></td>
							<td class="bg-green"><?php echo esc_html($successful_delivery_redx);?></td>
							<td class="bg-red"><?php echo esc_html($cancel_delivery_redx);?></td>
							<td><?php echo esc_html($success_ratio_redx);?>%</td>
						</tr>
						<tr>
							<td class="left col1">Paperfly</td>
							<td><?php echo esc_html($total_delivery_paperfly);?></td>
							<td class="bg-green"><?php echo esc_html($successful_delivery_paperfly);?></td>
							<td class="bg-red"><?php echo esc_html($cancel_delivery_paperfly);?></td>
							<td><?php echo esc_html($success_ratio_paperfly);?>%</td>
						</tr>
					</tbody>
					<tfoot>
						<tr>
							<th class="left col1">Total</th>
							<th><?php echo esc_html($total_delivery_total);?></th>
							<th><?php echo esc_html($successful_delivery_total);?></th>
							<th><?php echo esc_html($cancel_delivery_total);?></th>
							<th><?php echo esc_html($success_ratio_total);?>%</th>
						</tr>
					</tfoot>
				</table>
			</div>
			<div class="ogp-progress">
				<?php $failure_ratio_total = 100 - $success_ratio_total;?>
				<div class="ogp-progress-bar" style="width: <?php echo esc_html($success_ratio_total);?>%; background: #1cc88a;" role="progressbar">
					<p style="text-align: right; margin-right: 5px;"><?php echo esc_html($success_ratio_total) . '%';?></p>
				</div>
				<?php if($cancel_delivery_total > 0){?>
				<div class="ogp-progress-bar" style="width: <?php echo esc_html($failure_ratio_total);?>%; background: #e74a3b;" role="progressbar">
					<p style="text-align: left; margin-left: 5px;"><?php echo esc_html($failure_ratio_total) . '%';?></p>
				</div>
				<?php }?>
			</div>
		<?php
			}
		?>
	</div>
<?php	
}

function ogp_restrict_order_interval_by_mobile($mobile_number) {
	// translators: %d is the number of minutes the user must wait before placing another order.
    $ogp_order_interval_message = get_option('ogp_order_interval_message', __('⏳ পরবর্তী অর্ডারের জন্য আপনাকে %d মিনিট অপেক্ষা করতে হবে!', 'order-growth-plugin'));
    $ogp_order_interval = get_option('ogp_order_interval', 1);
    $ogp_order_interval_second = $ogp_order_interval * 60;
    
	if (empty($mobile_number)) {
        $mobile_number = WC()->checkout->get_value('billing_phone');
    }
	
    if (empty($mobile_number)) {
        return;
    }
    
    $args = array(
        'limit'     => -1,
        'billing_phone' => $mobile_number,
    );
    
    $orders = wc_get_orders($args);
    $current_time = current_time('timestamp');
    
    foreach ($orders as $order) {
		$order_time = $order->get_date_created()->getOffsetTimestamp();
        $time_diff = $current_time - $order_time;
        
        if ($time_diff < $ogp_order_interval_second) {
            $remaining_time = $ogp_order_interval_second - $time_diff;
            $time_diff_final = ceil($remaining_time / 60); 
            $message = sprintf($ogp_order_interval_message, $time_diff_final);
            wc_add_notice($message, 'error');
            return;
        }
		break;
    }
}

function ogp_get_user_ip() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return sanitize_text_field($_SERVER['HTTP_CLIENT_IP']);
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip_addresses = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        return sanitize_text_field(trim($ip_addresses[0]));
    } else {
        return sanitize_text_field($_SERVER['REMOTE_ADDR']);
    }
}

add_action('woocommerce_checkout_process', 'ogp_validate_process');
function ogp_validate_process() {
	$user_ip = ogp_get_user_ip(); 
	$ogp_order_interval_validation = get_option('ogp_order_interval_validation',1);
	$ogp_phone_validation = get_option('ogp_phone_validation',1);
    $ogp_phone_validation_message = get_option('ogp_phone_validation_message', '📱 অনুগ্রহ করে নিচে আপনার ১১ সংখ্যার সঠিক মোবাইল নাম্বারটি প্রদান করুন! 🔢 যেমন: 01XXXXXXXXX');
    $ogp_block_message = get_option('ogp_block_message', '🔒 আপনাকে কোনো কারণে আমাদের সিস্টেম থেকে ব্লক করা হয়েছে। 📞 অনুগ্রহ করে আমাদের সাথে যোগাযোগ করুন।');
	
	$checkout_fields = WC()->checkout()->get_checkout_fields();
	foreach ($checkout_fields['billing'] as $field_key => $field) {
        if (strpos($field_key, 'phone') !== false && isset($_POST[$field_key]) && !empty($_POST[$field_key])) {
            $billing_phone = $_POST[$field_key];
        }
    }
	
	if($ogp_order_interval_validation == 1){
		if (!empty($billing_phone)) {
			ogp_restrict_order_interval_by_mobile($billing_phone);
		}
	}
	
	if($ogp_phone_validation == 1){
		if (!empty($billing_phone)) {
			$billing_phone = preg_replace('/\D/', '', $_POST['billing_phone']); 
			if (strlen($billing_phone) != 11) {
				wc_add_notice(esc_attr($ogp_phone_validation_message), 'error');
			}
		}
	}
	
	if (isset($_POST['billing_phone']) && !empty($_POST['billing_phone'])) {
		if(ogp_exists_number($_POST['billing_phone'])) {
			ogp_block_count_time ('',$_POST['billing_phone']);
			wc_add_notice(esc_attr($ogp_block_message), 'error');
		};
	}
	
	if (!empty($user_ip)) {
		if(ogp_exists_ip($user_ip)) {
			ogp_block_count_time ($user_ip,'');
			wc_add_notice(esc_attr($ogp_block_message), 'error');
		}
	}
}

function ogp_block_count_time($ip_value = '', $number_value = '') {
	global $wpdb;
	$table_name = $wpdb->prefix . 'ogp_block7';
	
	if(!empty($number_value)) {
		$updated = $wpdb->query($wpdb->prepare(
			"UPDATE $table_name 
			SET blockedHits = blockedHits + 1, 
				lastAttempt = %d 
			WHERE phoneNumber = %s",
			current_time('timestamp'),
			$number_value
		));
	}
	
	if(!empty($ip_value)) {
		$updated = $wpdb->query($wpdb->prepare(
			"UPDATE $table_name 
			SET blockedHits = blockedHits + 1, 
				lastAttempt = %d 
			WHERE ip = %s",
			current_time('timestamp'),
			$ip_value
		));
	}
}

function ogp_exists_ip($ip_value) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';

    if (empty($ip_value)) {
        return false;
    }

    $ip_exists = (int) $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE ip = %s",
        $ip_value
    ));

    return $ip_exists > 0;
}

function ogp_exists_number($number_value) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';

    if (empty($number_value)) {
        return false;
    }

    $number_exists = (int) $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE phoneNumber = %s",
        $number_value
    ));

    return $number_exists > 0;
}

add_action('wp_ajax_ogp_check_and_block_ip_number', 'ogp_check_and_block_ip_number');
add_action('wp_ajax_nopriv_ogp_check_and_block_ip_number', 'ogp_check_and_block_ip_number');
function ogp_check_and_block_ip_number() { 
    if (!isset($_POST['ogp_blocking_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_nonce'], 'ogp_blocking')) {
        wp_send_json_error('Invalid nonce.');
        wp_die();
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';
    
    $block_type = isset($_POST['block_type']) ? sanitize_text_field($_POST['block_type']) : '';
    $ip_value = isset($_POST['ip_value']) ? sanitize_text_field($_POST['ip_value']) : '';
    $number_value = isset($_POST['number_value']) ? sanitize_text_field($_POST['number_value']) : '';
    $blocked_via_value = isset($_POST['ogp_blocked_via']) ? sanitize_text_field($_POST['ogp_blocked_via']) : '';

    if (ogp_exists_ip($ip_value) || ogp_exists_number($number_value)) {
        wp_send_json_error('Already exists in the block list.');
        wp_die();
    }

    $inserted = $wpdb->insert(
        $table_name,
        array(
            'type' => ($block_type === 'IP') ? 0 : 1,
            'ip' => $ip_value,
            'phoneNumber' => $number_value,
            'blockedTime' => current_time('timestamp'),
            'reason' => 'Blocked via ' . $blocked_via_value,
            'lastAttempt' => 0,
            'blockedHits' => 0,
            'expiration' => 0,
            'parameters' => '',
            'comment' => ''
        ),
        array('%d', '%s', '%s', '%d', '%s', '%d', '%d', '%d', '%s', '%s')
    );

    if ($inserted) {
        wp_send_json_success('Blocked successfully.');
    } else {
        wp_send_json_error('Failed to block.');
    }
    
    wp_die();
}

add_action('wp_ajax_ogp_check_and_unblock_ip_number', 'ogp_check_and_unblock_ip_number');
add_action('wp_ajax_nopriv_ogp_check_and_unblock_ip_number', 'ogp_check_and_unblock_ip_number');
function ogp_check_and_unblock_ip_number() { 
    if (!isset($_POST['ogp_blocking_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_nonce'], 'ogp_blocking')) {
        wp_send_json_error('Invalid nonce.');
        wp_die();
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';

    $block_type = isset($_POST['block_type']) ? sanitize_text_field($_POST['block_type']) : '';
    $ip_value = isset($_POST['ip_value']) ? sanitize_text_field($_POST['ip_value']) : '';
    $number_value = isset($_POST['number_value']) ? sanitize_text_field($_POST['number_value']) : '';

    // Check if the IP or Number exists before attempting deletion
    if ($block_type === 'IP' && ogp_exists_ip($ip_value)) {
        $deleted = $wpdb->delete($table_name, array('ip' => $ip_value), array('%s'));
        $message = 'IP unblocked successfully.';
    } elseif ($block_type === 'Number' && ogp_exists_number($number_value)) {
        $deleted = $wpdb->delete($table_name, array('phoneNumber' => $number_value), array('%s'));
        $message = 'Phone number unblocked successfully.';
    } else {
        wp_send_json_error(ucfirst($block_type) . ' not found in the block list.');
        wp_die();
    }

    if ($deleted) {
        wp_send_json_success($message);
    } else {
        wp_send_json_error('Failed to unblock.');
    }
    
    wp_die();
}

add_action('wp_ajax_ogp_search_block_number', 'ogp_search_block_number');
add_action('wp_ajax_nopriv_ogp_search_block_number', 'ogp_search_block_number');
function ogp_search_block_number() {
	if (!isset($_POST['ogp_blocking_search_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_search_nonce'], 'ogp_blocking_search')) {
        wp_send_json_error('Invalid nonce.');
        wp_die();
    }

	$ogp_blocking_number = isset($_POST['ogp_blocking_number']) ? sanitize_text_field($_POST['ogp_blocking_number']) : '';
    
	if (empty($ogp_blocking_number)) {
        wp_send_json_error(array('message' => 'No number provided'));
        wp_die();
    }

	global $wpdb;
	$table_name = $wpdb->prefix . 'ogp_block7';

	$row = $wpdb->get_row($wpdb->prepare(
		"SELECT * FROM $table_name WHERE phoneNumber = %s",
		$ogp_blocking_number
	));

	if (!empty($row)) {
		$block_type = ($row->type == 0) ? 'IP' : 'Number';
		$details = (!empty($row->ip)) ? $row->ip : $row->phoneNumber;
		$blocked_time_formatted = !empty($row->blockedTime) ? date('F j, Y g:i a', $row->blockedTime) : 'N/A';
		$blocked_reason = !empty($row->reason) ? $row->reason : 'N/A';
		$last_attempt_formatted = (!empty($row->lastAttempt) && $row->lastAttempt > 0) ? date('F j, Y g:i a', $row->lastAttempt) : 'N/A';

		$data = array(
		    'block_type' => $block_type,
		    'details' => $details,
		    'blocked_time' => $blocked_time_formatted,
		    'blocked_reason' => $blocked_reason,
		    'blocked_hits' => $row->blockedHits,
		    'last_attempt' => $last_attempt_formatted
		);
		
		wp_send_json_success($data);
	} else {
		wp_send_json_error(array('message' => 'No blocked records found.'));
	}
	wp_die();
}

add_action('wp_ajax_ogp_search_block_ip', 'ogp_search_block_ip');
add_action('wp_ajax_nopriv_ogp_search_block_ip', 'ogp_search_block_ip');
function ogp_search_block_ip() {
	if (!isset($_POST['ogp_blocking_search_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_search_nonce'], 'ogp_blocking_search')) {
        wp_send_json_error('Invalid nonce.');
        wp_die();
    }

	$ogp_blocking_ip = isset($_POST['ogp_blocking_ip']) ? sanitize_text_field($_POST['ogp_blocking_ip']) : '';
    
	if (empty($ogp_blocking_ip)) {
        wp_send_json_error(array('message' => 'No ip provided'));
        wp_die();
    }

	global $wpdb;
	$table_name = $wpdb->prefix . 'ogp_block7';

	$row = $wpdb->get_row($wpdb->prepare(
		"SELECT * FROM $table_name WHERE ip = %s",
		$ogp_blocking_ip
	));

	if (!empty($row)) {
		$block_type = ($row->type == 0) ? 'IP' : 'Number';
		$details = (!empty($row->ip)) ? $row->ip : $row->phoneNumber;
		$blocked_time_formatted = !empty($row->blockedTime) ? date('F j, Y g:i a', $row->blockedTime) : 'N/A';
		$blocked_reason = !empty($row->reason) ? $row->reason : 'N/A';
		$last_attempt_formatted = (!empty($row->lastAttempt) && $row->lastAttempt > 0) ? date('F j, Y g:i a', $row->lastAttempt) : 'N/A';

		$data = array(
		    'block_type' => $block_type,
		    'details' => $details,
		    'blocked_time' => $blocked_time_formatted,
		    'blocked_reason' => $blocked_reason,
		    'blocked_hits' => $row->blockedHits,
		    'last_attempt' => $last_attempt_formatted
		);
		
		wp_send_json_success($data);
	} else {
		wp_send_json_error(array('message' => 'No blocked records found.'));
	}
	wp_die();
}

add_action('wp_ajax_ogp_search_order', 'ogp_search_order');
add_action('wp_ajax_nopriv_ogp_search_order', 'ogp_search_order');
function ogp_search_order() {
    if (!isset($_POST['ogp_order_search_nonce']) || !wp_verify_nonce($_POST['ogp_order_search_nonce'], 'ogp_order_search')) {
        wp_send_json_error(array('message' => 'Invalid nonce.'));
        wp_die();
    }
    
    if (isset($_POST['ogp_order_number']) && !empty($_POST['ogp_order_number'])) {
        $order_number = sanitize_text_field($_POST['ogp_order_number']);
        $order = wc_get_order($order_number);
        
        if ($order) {
            $order_id = $order->get_id();
            $order_ip = $order->get_customer_ip_address();
            $billing_phone = $order->get_billing_phone();
            $order_date = $order->get_date_created()->date('M j, Y');
            $blocked_ip = ogp_block_ip_search($order_ip);
            $blocked_number = ogp_block_number_search($billing_phone);
            
            $response = array(
                'order_id' => $order_id,
                'order_ip' => $order_ip,
                'billing_phone' => $billing_phone,
                'order_date' => $order_date,
                'blocked_ip' => $blocked_ip,
                'blocked_number' => $blocked_number
            );
            
            wp_send_json_success($response);
        } else {
            wp_send_json_error(array('message' => 'No orders found.'));
        }
    } else {
        wp_send_json_error(array('message' => 'Invalid Order Number'));
    }

    wp_die();
}

function ogp_block_ip_search($ip) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';
    $exists = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE ip = %s",
        $ip
    ));
    return ($exists > 0) ? 1 : 0;
}

function ogp_block_number_search($number) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';
    $exists = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE phoneNumber = %s",
        $number
    ));
    return ($exists > 0) ? 1 : 0;
}

add_action('wp_ajax_ogp_update_block_comment', 'ogp_update_block_comment');
add_action('wp_ajax_nopriv_ogp_update_block_comment', 'ogp_update_block_comment');
function ogp_update_block_comment() {
    if (!isset($_POST['ogp_blocking_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_nonce'], 'ogp_blocking')) {
        wp_send_json_error('Invalid nonce.');
        wp_die();
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';
    
    $block_id = isset($_POST['block_id']) ? intval($_POST['block_id']) : 0;
    $comment = isset($_POST['comment']) ? sanitize_textarea_field($_POST['comment']) : '';
    
    if ($block_id <= 0) {
        wp_send_json_error('Invalid block ID.');
        wp_die();
    }
    
    $updated = $wpdb->update(
        $table_name,
        array('comment' => $comment),
        array('id' => $block_id),
        array('%s'),
        array('%d')
    );
    
    if ($updated !== false) {
        wp_send_json_success('Comment updated successfully.');
    } else {
        wp_send_json_error('Failed to update comment.');
    }
    
    wp_die();
}

add_action('wp_ajax_ogp_delete_block_entry', 'ogp_delete_block_entry');
add_action('wp_ajax_nopriv_ogp_delete_block_entry', 'ogp_delete_block_entry');
function ogp_delete_block_entry() {
    if (!isset($_POST['ogp_blocking_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_nonce'], 'ogp_blocking')) {
        wp_send_json_error('Invalid nonce.');
        wp_die();
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';
    
    $block_id = isset($_POST['block_id']) ? intval($_POST['block_id']) : 0;
    
    if ($block_id <= 0) {
        wp_send_json_error('Invalid block ID.');
        wp_die();
    }
    
    $block_entry = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_name WHERE id = %d",
        $block_id
    ));
    
    if (!$block_entry) {
        wp_send_json_error('Block entry not found.');
        wp_die();
    }
    
    $deleted = $wpdb->delete(
        $table_name,
        array('id' => $block_id),
        array('%d')
    );
    
    if ($deleted) {
        wp_send_json_success(array(
            'message' => 'Block entry deleted successfully.',
            'unblocked_ip' => $block_entry->ip,
            'unblocked_phone' => $block_entry->phoneNumber
        ));
    } else {
        wp_send_json_error('Failed to delete block entry.');
    }
    
    wp_die();
}

// Shortcode: [my_live_search items="01875838782,017XXXXXXXX" title="Optional Title"]
// If 'items' is omitted, it will pull distinct phone numbers and IPs from the OGP block table.
function ogp_my_live_search_shortcode($atts, $content = null){
    $atts = shortcode_atts(array(
        'items' => '',
        'title' => 'Live Search'
    ), $atts, 'my_live_search');

    $items = array();

    if (!empty($atts['items'])){
        // Allow comma or pipe separated items
        $raw = preg_split('/[|,]/', $atts['items']);
        foreach($raw as $r){
            $r = trim($r);
            if ($r !== '') $items[] = $r;
        }
    } else {
        global $wpdb;
        $table = $wpdb->prefix . 'ogp_block7';
        if ($wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table)) === $table){
            // Fetch last 200 items from phoneNumber and ip columns
            $phones = $wpdb->get_col("SELECT DISTINCT phoneNumber FROM {$table} WHERE phoneNumber <> '' ORDER BY blockedTime DESC LIMIT 200");
            $ips    = $wpdb->get_col("SELECT DISTINCT ip FROM {$table} WHERE ip <> '' ORDER BY blockedTime DESC LIMIT 200");
            $items = array_values(array_unique(array_merge($phones, $ips)));
        }
    }

    // Build markup
    ob_start();
    $uid = 'ogp_live_' . wp_generate_uuid4();
    echo '<div class="ogp-live-search" id="'.esc_attr($uid).'">';
    echo '<h3 style="margin:0 0 8px;">'.esc_html($atts['title']).'</h3>';
    echo '<input type="search" class="regular-text" placeholder="Start typing (e.g., 018, 103.)" style="max-width:520px;width:100%;">';
    echo '<ul style="margin-top:10px;padding-left:18px;">';
    if (!empty($items)){
        foreach($items as $item){
            $display = esc_html($item);
            $index   = preg_replace('/[^0-9a-z\.\+]/i','', strtolower($item));
            echo '<li data-ogp-item data-ogp-index="'.$index.'">'.$display.'</li>';
        }
    } else {
        echo '<li data-ogp-item data-ogp-index="sample">No data available. You can pass items via items="...".</li>';
    }
    echo '</ul>';
    echo '</div>';
    return ob_get_clean();
}
add_shortcode('my_live_search', 'ogp_my_live_search_shortcode');

/**
 * Lookup the latest block entry ID for a given phone/ip pair.
 * This lets the Block List delete instantly after a fresh "Block" without page refresh.
 */
add_action('wp_ajax_ogp_lookup_block_id', 'ogp_lookup_block_id');
add_action('wp_ajax_nopriv_ogp_lookup_block_id', 'ogp_lookup_block_id');
function ogp_lookup_block_id(){
    if (!isset($_POST['ogp_blocking_nonce']) || !wp_verify_nonce($_POST['ogp_blocking_nonce'], 'ogp_blocking')) {
        wp_send_json_error(array('message' => 'Invalid nonce.'));
        wp_die();
    }
    global $wpdb;
    $table_name = $wpdb->prefix . 'ogp_block7';
    $ip = isset($_POST['ip_value']) ? sanitize_text_field($_POST['ip_value']) : '';
    $phone = isset($_POST['number_value']) ? sanitize_text_field($_POST['number_value']) : '';
    $where = array();
    $params = array();
    if ($ip !== '') { $where[] = "ip = %s"; $params[] = $ip; }
    if ($phone !== '') { $where[] = "phoneNumber = %s"; $params[] = $phone; }
    if (empty($where)) {
        wp_send_json_error(array('message' => 'Missing lookup parameters.'));
        wp_die();
    }
    $sql = "SELECT id, ip, phoneNumber, created_at FROM {$table_name} WHERE " . implode(" AND ", $where) . " ORDER BY id DESC LIMIT 1";
    $prepared = $wpdb->prepare($sql, $params);
    $row = $wpdb->get_row($prepared, ARRAY_A);
    if ($row) {
        wp_send_json_success(array('id' => intval($row['id']), 'ip' => $row['ip'], 'phone' => $row['phoneNumber'], 'created_at' => $row['created_at']));
    } else {
        wp_send_json_error(array('message' => 'Not found.'));
    }
    wp_die();
}
?>
